import {AdminDocsUrls} from '@app/admin/admin-config';
import {AdminSettings} from '@common/admin/settings/admin-settings';
import {JsonChipField} from '@common/admin/settings/layout/json-chip-field';
import {SettingsErrorGroup} from '@common/admin/settings/layout/settings-error-group';
import {AdminSettingsLayout} from '@common/admin/settings/layout/settings-layout';
import {
  SettingsPanel,
  SettingsSectionHeader,
} from '@common/admin/settings/layout/settings-panel';
import {supportedBackends} from '@common/admin/settings/pages/uploading-settings/backends/backends';
import {BackendsSection} from '@common/admin/settings/pages/uploading-settings/backends/backends-section';
import {useMaxServerUploadSize} from '@common/admin/settings/pages/uploading-settings/max-server-upload-size';
import {useAdminSettings} from '@common/admin/settings/requests/use-admin-settings';
import {SectionHelper} from '@common/ui/other/section-helper';
import {FormFileSizeField} from '@common/uploads/components/file-size-field';
import {Accordion, AccordionItem} from '@ui/accordion/accordion';
import {FormChipField} from '@ui/forms/input-field/chip-field/form-chip-field';
import {FormTextField} from '@ui/forms/input-field/text-field/text-field';
import {Item} from '@ui/forms/listbox/item';
import {FormRadio} from '@ui/forms/radio-group/radio';
import {FormRadioGroup} from '@ui/forms/radio-group/radio-group';
import {message} from '@ui/i18n/message';
import {Trans} from '@ui/i18n/trans';
import {useTrans} from '@ui/i18n/use-trans';
import {Fragment, useState} from 'react';
import {useForm, useWatch} from 'react-hook-form';

// todo: test digital ocean/dropbox/backblaze types

export const uploadTypes = [
  {
    name: 'bedrive',
    label: 'Bedrive uploads',
  },
  {
    name: 'articles',
    label: 'Article uploads',
  },
  {
    name: 'avatars',
    label: 'User avatars',
  },
  {
    name: 'branding',
    label: 'Branding',
  },
];

export function Component() {
  const {data} = useAdminSettings();
  const form = useForm<AdminSettings>({
    defaultValues: {
      client: {
        uploading: data.client.uploading ?? {},
        uploads: {
          chunk_size: data.client.uploads.chunk_size ?? 5242880,
        },
      },
      server: {
        static_file_delivery: data.server.static_file_delivery ?? '',
      },
    },
  });
  return (
    <AdminSettingsLayout
      form={form}
      title={<Trans message="Uploading" />}
      docsLink={AdminDocsUrls.settings.uploading}
    >
      <BackendsSection />
      <UploadTypesSection />
      <GlobalSettingsPanel />
    </AdminSettingsLayout>
  );
}

function UploadTypesSection() {
  const [expandedValues, setExpandedValues] = useState<number[]>([]);

  return (
    <Fragment>
      <SettingsSectionHeader margin="mt-44 mb-12" size="lg">
        <Trans message="Upload types" />
        <Trans message="Configure storage methods and file restrictions for different upload types accross the site." />
      </SettingsSectionHeader>
      <Accordion
        variant="outline"
        mode="multiple"
        expandedValues={expandedValues}
        onExpandedChange={values => setExpandedValues(values as number[])}
        size="lg"
      >
        {uploadTypes.map(type => (
          <AccordionItem
            bodyPadding="p-24"
            key={type.name}
            label={<Trans message={type.label} />}
          >
            <UploadTypeFields name={type.name} />
          </AccordionItem>
        ))}
      </Accordion>
    </Fragment>
  );
}

type UploadTypeFieldsProps = {
  name: string;
};
function UploadTypeFields({name}: UploadTypeFieldsProps) {
  return (
    <div className="space-y-20">
      <BackendsField name={name} />
      <FormTextField
        name={`client.uploading.types.${name}.root`}
        label={<Trans message="Folder" />}
        description={
          <Trans message="Where in the backend uploads should be stored. Relative to backend root. Leave empty to store at root." />
        }
      />
      <MaxUploadSizeField name={name} />
      <AvailableSpaceField name={name} />
      <AllowedExtensionsField name={name} />
      <BlockedExtensionsField name={name} />
    </div>
  );
}

type BackendsFieldProps = {
  name: string;
};
function BackendsField({name}: BackendsFieldProps) {
  const availableBackends = useWatch<
    AdminSettings,
    'client.uploading.backends'
  >({
    name: 'client.uploading.backends',
  });
  return (
    <FormChipField
      label={<Trans message="Storage backends" />}
      name={`client.uploading.types.${name}.backends`}
      suggestions={availableBackends}
      valueKey="id"
      description={
        <Trans message="Select in which backend(s) these uploads should be stored." />
      }
    >
      {backend => {
        const config = supportedBackends.find(b => b.type === backend.type);
        return (
          <Item key={backend.id} value={backend.id}>
            {config?.label ? <Trans {...config.label} /> : backend.type}
          </Item>
        );
      }}
    </FormChipField>
  );
}

function GlobalSettingsPanel() {
  return (
    <div>
      <SettingsSectionHeader margin="mt-44 mb-12" size="lg">
        <Trans message="Global settings" />
        <Trans message="Settings that will apply to all upload types and backends." />
      </SettingsSectionHeader>
      <FileDeliveryPanel />
      <ChunkSizePanel />
    </div>
  );
}

function FileDeliveryPanel() {
  return (
    <SettingsPanel
      className="mb-24"
      title={<Trans message="File Delivery Optimization" />}
      description={
        <Trans message="Both X-Sendfile and X-Accel need to be enabled on the server first. When enabled, it will reduce server memory and CPU usage when previewing or downloading files, especially for large files." />
      }
    >
      <SettingsErrorGroup
        name="static_delivery_group"
        separatorBottom={false}
        separatorTop={false}
      >
        {isInvalid => (
          <FormRadioGroup
            invalid={isInvalid}
            size="sm"
            name="server.static_file_delivery"
            orientation="vertical"
          >
            <FormRadio value="">
              <Trans message="None" />
            </FormRadio>
            <FormRadio value="xsendfile">
              <Trans message="X-Sendfile (Apache)" />
            </FormRadio>
            <FormRadio value="xaccel">
              <Trans message="X-Accel (Nginx)" />
            </FormRadio>
          </FormRadioGroup>
        )}
      </SettingsErrorGroup>
    </SettingsPanel>
  );
}

function ChunkSizePanel() {
  const {data} = useMaxServerUploadSize();
  return (
    <SettingsPanel
      className="mb-24"
      title={<Trans message="Chunk Size" />}
      description={
        <Trans message="Upload larger files in specified size chunks. This allows for resuming failed uploads and bypassing maximum file size limits on the server." />
      }
    >
      <FormFileSizeField size="sm" name="client.uploads.chunk_size" />
      <SectionHelper
        className="mt-12"
        color="warning"
        size="sm"
        description={
          <Trans
            message="Maximum upload size on your server currently is set to <b>:size</b>"
            values={{size: data?.maxSize, b: chunks => <b>{chunks}</b>}}
          />
        }
      />
    </SettingsPanel>
  );
}

type MaxUploadSizeFieldProps = {
  name: string;
};
export function MaxUploadSizeField({name}: MaxUploadSizeFieldProps) {
  return (
    <FormFileSizeField
      min={1}
      descriptionPosition="top"
      label={<Trans message="Maximum file size" />}
      description={
        <Trans message="Size (in bytes) for a single file user can upload." />
      }
      name={`client.uploading.types.${name}.max_file_size`}
    />
  );
}

type AvailableSpaceFieldProps = {
  name: string;
};
export function AvailableSpaceField({name}: AvailableSpaceFieldProps) {
  return (
    <FormFileSizeField
      min={1}
      descriptionPosition="top"
      label={<Trans message="Available space" />}
      description={
        <Trans message="Disk space (in bytes) each user uploads are allowed to take up." />
      }
      name={`client.uploading.types.${name}.available_space`}
    />
  );
}

type AllowedExtensionsFieldProps = {
  name: string;
};
export function AllowedExtensionsField({name}: AllowedExtensionsFieldProps) {
  const {trans} = useTrans();
  return (
    <JsonChipField
      name={`client.uploading.types.${name}.allowed_extensions`}
      placeholder={trans(message('Add extension...'))}
      label={<Trans message="Allowed extensions" />}
      descriptionPosition="top"
      description={
        <Trans message="List of allowed file types (jpg, mp3, pdf etc.). Leave empty to allow all file types." />
      }
    />
  );
}

type BlockedExtensionsFieldProps = {
  name: string;
};
export function BlockedExtensionsField({name}: BlockedExtensionsFieldProps) {
  const {trans} = useTrans();

  return (
    <JsonChipField
      name={`client.uploading.types.${name}.blocked_extensions`}
      placeholder={trans(message('Add extension...'))}
      label={<Trans message="Blocked extensions" />}
      descriptionPosition="top"
      description={
        <Trans message="Prevent uploading of these file types, even if they are allowed above." />
      }
    />
  );
}
