import {AdminDocsUrls} from '@app/admin/admin-config';
import {AdminSettings} from '@common/admin/settings/admin-settings';
import {JsonChipField} from '@common/admin/settings/layout/json-chip-field';
import {SettingsErrorGroup} from '@common/admin/settings/layout/settings-error-group';
import {AdminSettingsLayout} from '@common/admin/settings/layout/settings-layout';
import {DocsLink} from '@common/admin/settings/layout/settings-links';
import {SettingsPanel} from '@common/admin/settings/layout/settings-panel';
import {DropboxForm} from '@common/admin/settings/pages/uploading-settings/dropbox-form/dropbox-form';
import {useMaxServerUploadSize} from '@common/admin/settings/pages/uploading-settings/max-server-upload-size';
import {useUploadS3Cors} from '@common/admin/settings/pages/uploading-settings/use-upload-s3-cors';
import {useAdminSettings} from '@common/admin/settings/requests/use-admin-settings';
import {SectionHelper} from '@common/ui/other/section-helper';
import {FormFileSizeField} from '@common/uploads/components/file-size-field';
import {Button} from '@ui/buttons/button';
import {FormTextField} from '@ui/forms/input-field/text-field/text-field';
import {Item} from '@ui/forms/listbox/item';
import {FormRadio} from '@ui/forms/radio-group/radio';
import {FormRadioGroup} from '@ui/forms/radio-group/radio-group';
import {FormSelect} from '@ui/forms/select/select';
import {FormSwitch} from '@ui/forms/toggle/switch';
import {message} from '@ui/i18n/message';
import {Trans} from '@ui/i18n/trans';
import {useTrans} from '@ui/i18n/use-trans';
import {Fragment} from 'react';
import {useForm, useFormContext} from 'react-hook-form';

export function Component() {
  const {data} = useAdminSettings();
  const form = useForm<AdminSettings>({
    defaultValues: {
      client: {
        uploads: {
          max_size: data.client.uploads.max_size ?? 52428800,
          chunk_size: data.client.uploads.chunk_size ?? 5242880,
          available_space: data.client.uploads.available_space ?? 104857600,
          allowed_extensions: data.client.uploads.allowed_extensions ?? [],
          blocked_extensions: data.client.uploads.blocked_extensions ?? [],
          s3_direct_upload: data.client.uploads.s3_direct_upload ?? false,
        },
      },
      server: {
        static_file_delivery: data.server.static_file_delivery ?? '',
        uploads_disk_driver: data.server.uploads_disk_driver ?? 'local',
        public_disk_driver: data.server.public_disk_driver ?? 'local',

        // s3
        storage_s3_key: data.server.storage_s3_key ?? '',
        storage_s3_secret: data.server.storage_s3_secret ?? '',
        storage_s3_region: data.server.storage_s3_region ?? '',
        storage_s3_bucket: data.server.storage_s3_bucket ?? '',
        storage_s3_endpoint: data.server.storage_s3_endpoint ?? '',

        // digitalocean spaces
        storage_digitalocean_key: data.server.storage_digitalocean_key ?? '',
        storage_digitalocean_secret:
          data.server.storage_digitalocean_secret ?? '',
        storage_digitalocean_region:
          data.server.storage_digitalocean_region ?? '',
        storage_digitalocean_bucket:
          data.server.storage_digitalocean_bucket ?? '',

        // backblaze
        storage_backblaze_key: data.server.storage_backblaze_key ?? '',
        storage_backblaze_secret: data.server.storage_backblaze_secret ?? '',
        storage_backblaze_region: data.server.storage_backblaze_region ?? '',
        storage_backblaze_bucket: data.server.storage_backblaze_bucket ?? '',

        // ftp
        storage_ftp_host: data.server.storage_ftp_host ?? '',
        storage_ftp_username: data.server.storage_ftp_username ?? '',
        storage_ftp_password: data.server.storage_ftp_password ?? '',
        storage_ftp_root: data.server.storage_ftp_root ?? '',
        storage_ftp_port: data.server.storage_ftp_port ?? '21',
        storage_ftp_passive: data.server.storage_ftp_passive ?? false,
        storage_ftp_ssl: data.server.storage_ftp_ssl ?? false,

        // dropbox
        storage_dropbox_app_key: data.server.storage_dropbox_app_key ?? '',
        storage_dropbox_app_secret:
          data.server.storage_dropbox_app_secret ?? '',
        storage_dropbox_refresh_token:
          data.server.storage_dropbox_refresh_token ?? '',
      },
    },
  });
  return (
    <AdminSettingsLayout
      form={form}
      title={<Trans message="Uploading" />}
      docsLink={AdminDocsUrls.settings.uploading}
    >
      <StorageMethodsPanel />
      <CredentialsSection />
      <DirectUploadPanel />
      <FileDeliveryPanel />
      <ChunkSizePanel />
      <MaxUploadSizePanel />
      <FileSizeLimitsPanel />
      <AllowedExtensionsPanel />
      <BlockedExtensionsPanel />
    </AdminSettingsLayout>
  );
}

function StorageMethodsPanel() {
  return (
    <SettingsPanel
      className="mb-24"
      title={<Trans message="Storage Methods" />}
      description={
        <Trans message="Configure where user uploads should be stored." />
      }
    >
      <PrivateUploadField />
      <PublicUploadField />
    </SettingsPanel>
  );
}

function PrivateUploadField() {
  const {watch, clearErrors} = useFormContext<AdminSettings>();
  const isEnabled = watch('server.uploads_disk_driver');

  if (!isEnabled) return null;

  return (
    <FormSelect
      size="sm"
      className="mb-20"
      selectionMode="single"
      name="server.uploads_disk_driver"
      label={<Trans message="User Uploads Storage Method" />}
      description={
        <Trans message="Where should user private file uploads be stored." />
      }
      onSelectionChange={() => {
        clearErrors();
      }}
    >
      <Item value="local">
        <Trans message="Local Disk (Default)" />
      </Item>
      <Item value="ftp">FTP</Item>
      <Item value="digitalocean_s3">DigitalOcean Spaces</Item>
      <Item value="backblaze_s3">Backblaze</Item>
      <Item value="s3">Amazon S3 (Or compatible service)</Item>
      <Item value="dropbox">Dropbox</Item>
      <Item value="rackspace">Rackspace</Item>
    </FormSelect>
  );
}

function PublicUploadField() {
  const {watch, clearErrors} = useFormContext<AdminSettings>();
  const isEnabled = watch('server.public_disk_driver');

  if (!isEnabled) return null;

  return (
    <FormSelect
      size="sm"
      label={<Trans message="Public Uploads Storage Method" />}
      selectionMode="single"
      name="server.public_disk_driver"
      description={
        <Trans message="Where should user public uploads (like avatars) be stored." />
      }
      onSelectionChange={() => {
        clearErrors();
      }}
    >
      <Item value="local">
        <Trans message="Local Disk (Default)" />
      </Item>
      <Item value="s3">Amazon S3</Item>
      <Item value="ftp">FTP</Item>
      <Item value="digitalocean_s3">DigitalOcean Spaces</Item>
      <Item value="backblaze_s3">Backblaze</Item>
    </FormSelect>
  );
}

function FileDeliveryPanel() {
  return (
    <SettingsPanel
      className="mb-24"
      title={<Trans message="File Delivery Optimization" />}
      description={
        <Trans message="Both X-Sendfile and X-Accel need to be enabled on the server first. When enabled, it will reduce server memory and CPU usage when previewing or downloading files, especially for large files." />
      }
    >
      <SettingsErrorGroup
        name="static_delivery_group"
        separatorBottom={false}
        separatorTop={false}
      >
        {isInvalid => (
          <FormRadioGroup
            invalid={isInvalid}
            size="sm"
            name="server.static_file_delivery"
            orientation="vertical"
          >
            <FormRadio value="">
              <Trans message="None" />
            </FormRadio>
            <FormRadio value="xsendfile">
              <Trans message="X-Sendfile (Apache)" />
            </FormRadio>
            <FormRadio value="xaccel">
              <Trans message="X-Accel (Nginx)" />
            </FormRadio>
          </FormRadioGroup>
        )}
      </SettingsErrorGroup>
    </SettingsPanel>
  );
}

function ChunkSizePanel() {
  const {data} = useMaxServerUploadSize();
  return (
    <SettingsPanel
      className="mb-24"
      title={<Trans message="Chunk Size" />}
      description={
        <Trans message="Upload larger files in specified size chunks. This allows for resuming failed uploads and bypassing maximum file size limits on the server." />
      }
    >
      <FormFileSizeField size="sm" name="client.uploads.chunk_size" />
      <SectionHelper
        className="mt-12"
        color="warning"
        size="sm"
        description={
          <Trans
            message="Maximum upload size on your server currently is set to <b>:size</b>"
            values={{size: data?.maxSize, b: chunks => <b>{chunks}</b>}}
          />
        }
      />
    </SettingsPanel>
  );
}

function MaxUploadSizePanel() {
  return (
    <SettingsPanel
      className="mb-24"
      title={<Trans message="Maximum File Size" />}
      description={
        <Trans message="Maximum size (in bytes) for a single file user can upload." />
      }
    >
      <FormFileSizeField size="sm" min={1} name="client.uploads.max_size" />
    </SettingsPanel>
  );
}

function FileSizeLimitsPanel() {
  return (
    <SettingsPanel
      className="mb-24"
      title={<Trans message="Available Space" />}
      description={
        <Trans message="Disk space (in bytes) each user uploads are allowed to take up. This can be overridden per user." />
      }
    >
      <FormFileSizeField
        size="sm"
        min={1}
        name="client.uploads.available_space"
      />
    </SettingsPanel>
  );
}

function AllowedExtensionsPanel() {
  const {trans} = useTrans();
  return (
    <SettingsPanel
      className="mb-24"
      title={<Trans message="Allowed Extensions" />}
      description={
        <Trans message="List of allowed file types (jpg, mp3, pdf etc.). Leave empty to allow all file types." />
      }
    >
      <JsonChipField
        size="sm"
        name="client.uploads.allowed_extensions"
        placeholder={trans(message('Add extension...'))}
      />
    </SettingsPanel>
  );
}

function BlockedExtensionsPanel() {
  const {trans} = useTrans();
  return (
    <SettingsPanel
      title={<Trans message="Blocked Extensions" />}
      description={
        <Trans message="Prevent uploading of these file types, even if they are allowed above." />
      }
    >
      <JsonChipField
        size="sm"
        name="client.uploads.blocked_extensions"
        placeholder={trans(message('Add extension...'))}
      />
    </SettingsPanel>
  );
}

function CredentialsSection() {
  const {watch} = useFormContext<AdminSettings>();
  const drives = [
    watch('server.uploads_disk_driver'),
    watch('server.public_disk_driver'),
  ];

  if (drives.every(d => !d || d === 'local')) {
    return null;
  }

  let storageProvider = '';
  let docsLink = '';
  if (drives.includes('s3')) {
    storageProvider = 'Amazon S3';
    docsLink =
      'https://support.vebto.com/hc/articles/21/25/216/storing-files-on-amazon-s3';
  } else if (drives.includes('ftp')) {
    storageProvider = 'FTP';
  } else if (drives.includes('dropbox')) {
    storageProvider = 'Dropbox';
    docsLink =
      'https://support.vebto.com/hc/articles/21/25/215/storing-files-on-dropbox';
  } else if (drives.includes('digitalocean_s3')) {
    storageProvider = 'DigitalOcean Spaces';
  } else if (drives.includes('backblaze_s3')) {
    storageProvider = 'Backblaze';
    docsLink =
      'https://support.vebto.com/hc/articles/21/25/217/storing-files-on-backblaze';
  }

  return (
    <SettingsPanel
      className="mb-24"
      title={
        <Trans
          message=":provider Credentials"
          values={{provider: storageProvider}}
        />
      }
      description={
        <Trans message="Enter credentials for connecting to your storage provider." />
      }
      link={docsLink ? <DocsLink link={docsLink} /> : undefined}
    >
      <SettingsErrorGroup
        name="storage_group"
        separatorBottom={false}
        separatorTop={false}
      >
        {isInvalid => {
          if (drives.includes('s3')) {
            return <S3Form isInvalid={isInvalid} />;
          }
          if (drives.includes('ftp')) {
            return <FtpForm isInvalid={isInvalid} />;
          }
          if (drives.includes('dropbox')) {
            return <DropboxForm isInvalid={isInvalid} />;
          }
          if (drives.includes('digitalocean_s3')) {
            return <DigitalOceanForm isInvalid={isInvalid} />;
          }
          if (drives.includes('backblaze_s3')) {
            return <BackblazeForm isInvalid={isInvalid} />;
          }
        }}
      </SettingsErrorGroup>
    </SettingsPanel>
  );
}

export interface CredentialFormProps {
  isInvalid: boolean;
}
function DirectUploadPanel() {
  const uploadCors = useUploadS3Cors();
  const {data: defaultSettings} = useAdminSettings();

  const {watch} = useFormContext<AdminSettings>();
  const drives = [
    watch('server.uploads_disk_driver'),
    watch('server.public_disk_driver'),
  ];

  const s3DriverSelected = drives.some(d => d?.endsWith('s3'));
  if (!s3DriverSelected) return null;

  const s3DriverEnabled =
    defaultSettings?.server.uploads_disk_driver?.endsWith('s3') ||
    defaultSettings?.server.public_disk_driver?.endsWith('s3');

  return (
    <SettingsPanel
      className="mb-24"
      title={<Trans message="Direct Upload" />}
      description={
        <Fragment>
          <p>
            <Trans message="Upload files directly from browser to cloud storage, bypassing the server. This improves upload speeds and reduces server bandwidth usage." />
          </p>
          <p className="mt-10">
            <Trans message="Only disable this if your storage provider doesn't support multipart uploads." />
          </p>
        </Fragment>
      }
    >
      <FormSwitch
        size="sm"
        name="client.uploads.s3_direct_upload"
        description={
          <Trans message="If storage provider is not configured to allow browser uploads, use the CORS button below after saving credentials." />
        }
      >
        <Trans message="Enable direct upload" />
      </FormSwitch>
      <Button
        variant="flat"
        color="primary"
        size="xs"
        className="mt-20"
        onClick={() => {
          uploadCors.mutate();
        }}
        disabled={!s3DriverEnabled || uploadCors.isPending}
      >
        <Trans message="Configure CORS" />
      </Button>
    </SettingsPanel>
  );
}

function S3Form({isInvalid}: CredentialFormProps) {
  return (
    <Fragment>
      <FormTextField
        size="sm"
        invalid={isInvalid}
        className="mb-20"
        name="server.storage_s3_key"
        label={<Trans message="Amazon S3 key" />}
        required
      />
      <FormTextField
        size="sm"
        invalid={isInvalid}
        className="mb-20"
        name="server.storage_s3_secret"
        label={<Trans message="Amazon S3 secret" />}
        required
      />
      <FormTextField
        size="sm"
        invalid={isInvalid}
        className="mb-20"
        name="server.storage_s3_region"
        label={<Trans message="Amazon S3 region" />}
        placeholder="us-east-1"
      />
      <FormTextField
        size="sm"
        invalid={isInvalid}
        className="mb-20"
        name="server.storage_s3_bucket"
        label={<Trans message="Amazon S3 bucket" />}
        required
      />
      <FormTextField
        size="sm"
        invalid={isInvalid}
        name="server.storage_s3_endpoint"
        label={<Trans message="Amazon S3 endpoint" />}
        description={
          <Trans message="Only change endpoint if you are using another S3 compatible storage service." />
        }
      />
    </Fragment>
  );
}

function DigitalOceanForm({isInvalid}: CredentialFormProps) {
  return (
    <Fragment>
      <FormTextField
        size="sm"
        invalid={isInvalid}
        className="mb-20"
        name="server.storage_digitalocean_key"
        label={<Trans message="DigitalOcean key" />}
        required
      />
      <FormTextField
        size="sm"
        invalid={isInvalid}
        className="mb-20"
        name="server.storage_digitalocean_secret"
        label={<Trans message="DigitalOcean secret" />}
        required
      />
      <FormTextField
        size="sm"
        invalid={isInvalid}
        className="mb-20"
        name="server.storage_digitalocean_region"
        label={<Trans message="DigitalOcean region" />}
        pattern="[a-z0-9\-]+"
        placeholder="nyc3"
        required
      />
      <FormTextField
        size="sm"
        invalid={isInvalid}
        className="mb-20"
        name="server.storage_digitalocean_bucket"
        label={<Trans message="DigitalOcean bucket" />}
        required
      />
    </Fragment>
  );
}

function BackblazeForm({isInvalid}: CredentialFormProps) {
  return (
    <Fragment>
      <FormTextField
        size="sm"
        invalid={isInvalid}
        className="mb-20"
        name="server.storage_backblaze_key"
        label={<Trans message="Backblaze KeyID" />}
        required
      />
      <FormTextField
        size="sm"
        invalid={isInvalid}
        className="mb-20"
        name="server.storage_backblaze_secret"
        label={<Trans message="Backblaze applicationKey" />}
        required
      />
      <FormTextField
        size="sm"
        invalid={isInvalid}
        className="mb-20"
        name="server.storage_backblaze_region"
        label={<Trans message="Backblaze Region" />}
        pattern="[a-z0-9\-]+"
        placeholder="us-west-002"
        required
      />
      <FormTextField
        size="sm"
        invalid={isInvalid}
        className="mb-20"
        name="server.storage_backblaze_bucket"
        label={<Trans message="Backblaze bucket name" />}
        required
      />
    </Fragment>
  );
}

function FtpForm({isInvalid}: CredentialFormProps) {
  return (
    <>
      <FormTextField
        invalid={isInvalid}
        size="sm"
        className="mb-20"
        name="server.storage_ftp_host"
        label={<Trans message="FTP hostname" />}
        required
      />
      <FormTextField
        invalid={isInvalid}
        size="sm"
        className="mb-20"
        name="server.storage_ftp_username"
        label={<Trans message="FTP username" />}
        required
      />
      <FormTextField
        invalid={isInvalid}
        size="sm"
        className="mb-20"
        name="server.storage_ftp_password"
        label={<Trans message="FTP password" />}
        type="password"
        required
      />
      <FormTextField
        invalid={isInvalid}
        size="sm"
        className="mb-20"
        name="server.storage_ftp_root"
        label={<Trans message="FTP directory" />}
        placeholder="/"
      />
      <FormTextField
        invalid={isInvalid}
        size="sm"
        className="mb-20"
        name="server.storage_ftp_port"
        label={<Trans message="FTP port" />}
        type="number"
        min={0}
        placeholder="21"
      />
      <FormSwitch
        invalid={isInvalid}
        name="server.storage_ftp_passive"
        className="mb-10"
      >
        <Trans message="Passive" />
      </FormSwitch>
      <FormSwitch invalid={isInvalid} name="server.storage_ftp_ssl">
        <Trans message="SSL" />
      </FormSwitch>
    </>
  );
}
