import {message} from '@ui/i18n/message';
import {useTrans} from '@ui/i18n/use-trans';
import {useMemo} from 'react';
import {UseControllerProps} from 'react-hook-form';

type Props = {
  name: string;
  errorMessageName?: string;
  required?: boolean;
  minLength?: number;
  maxLength?: number;
  min?: number | string;
  max?: number | string;
  pattern?: string;
  type?: string;
};
export function useDefaultValidationRules({
  name,
  errorMessageName,
  required,
  minLength,
  maxLength,
  min,
  max,
  pattern,
  type,
}: Props) {
  const {trans} = useTrans();

  return useMemo(() => {
    const rules: UseControllerProps['rules'] = {};

    const attribute = trans(message(errorMessageName ?? name));

    if (required != null) {
      rules.required = {
        value: required,
        message: trans(
          message('The :attribute field is required.', {
            values: {attribute},
          }),
        ),
      };
    }

    if (minLength != null) {
      rules.minLength = {
        value: minLength,
        message: trans(
          message('The :attribute must be at least :min characters.', {
            values: {attribute, min: minLength},
          }),
        ),
      };
    }

    if (maxLength != null) {
      rules.maxLength = {
        value: maxLength,
        message: trans(
          message('The :attribute may not be greater than :max characters.', {
            values: {attribute, max: maxLength},
          }),
        ),
      };
    }

    if (min != null) {
      rules.min = {
        value: min,
        message: trans(
          message('The :attribute must be at least :min.', {
            values: {attribute, min},
          }),
        ),
      };
    }

    if (max != null) {
      rules.max = {
        value: max,
        message: trans(
          message('The :attribute may not be greater than :max.', {
            values: {attribute, max},
          }),
        ),
      };
    }

    if (pattern != null) {
      rules.pattern = {
        value: new RegExp(pattern),
        message: trans(
          message('The :attribute format is invalid.', {
            values: {attribute},
          }),
        ),
      };
    }

    rules.validate = (value: string) => {
      if (!value) {
        return;
      }
      if (type == 'url') {
        try {
          new URL(value);
        } catch (e) {
          return trans(
            message('This URL is invalid.', {
              values: {attribute},
            }),
          );
        }
      }

      if (type == 'email') {
        const emailRegex =
          /^(([^<>()[\]\\.,;:\s@"]+(\.[^<>()[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
        if (!emailRegex.test(value)) {
          return trans(
            message('This email is invalid.', {
              values: {attribute},
            }),
          );
        }
      }
    };

    return rules;
  }, [name, required, minLength, maxLength, min, max, pattern]);
}
