<?php

namespace Common\Files\Actions;

use Common\Files\UploadDiskResolver;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;

class FileUploadValidator
{
    public function __construct(
        protected array|null $allowedExtensions = null,
        protected array|null $blockedExtensions = null,
        protected int|null $maxFileSize = null,
        protected int|null $usedSpace = null,
        protected int|null $availableSpace = null,
    ) {}

    public static function validateForUploadType(
        string $uploadType,
        int $fileSize,
        string $extension,
        string $mime,
    ): Collection|null {
        $uploadTypeConfig = UploadDiskResolver::getUploadTypeConfig(
            $uploadType,
        );
        // todo: after max allowed space refactor check if max allowed space is available for this upload type and if user has that restrictions before loading this and validating
        $spaceUsage = (new GetUserSpaceUsage())->execute(
            uploadType: $uploadType,
        );
        return (new self(
            allowedExtensions: Arr::get(
                $uploadTypeConfig,
                'allowed_extensions',
            ),
            blockedExtensions: Arr::get(
                $uploadTypeConfig,
                'blocked_extensions',
            ),
            maxFileSize: Arr::get($uploadTypeConfig, 'max_file_size'),
            usedSpace: $spaceUsage['used'],
            availableSpace: $spaceUsage['available'],
        ))->validate(fileSize: $fileSize, extension: $extension, mime: $mime);
    }

    public function validate(
        int $fileSize,
        string $extension,
        string $mime,
    ): Collection|null {
        $errors = collect([
            'size' => $this->validateMaximumFileSize($fileSize),
            'spaceUsage' => $this->validateAllowedStorageSpace($fileSize),
            'allowedExtensions' => $this->validateAllowedExtensions($extension),
            'blockedExtensions' => $this->validateBlockedExtensions(
                $extension,
                $mime,
            ),
        ])->filter(fn($msg) => !is_null($msg));

        if (!$errors->isEmpty()) {
            return $errors;
        }

        return null;
    }

    protected function validateMaximumFileSize(?int $fileSize = null): ?string
    {
        if (is_null($this->maxFileSize) || is_null($fileSize)) {
            return null;
        }

        if ($fileSize > $this->maxFileSize) {
            return __('The file size may not be greater than :size', [
                'size' => self::formatBytes($this->maxFileSize),
            ]);
        }

        return null;
    }

    protected function validateAllowedStorageSpace(
        ?int $fileSize = null,
    ): string|null {
        if (is_null($fileSize) || is_null($this->availableSpace)) {
            return null;
        }

        $usedSpace = $this->usedSpace ?? 0;
        $enoughSpace = $usedSpace + $fileSize <= $this->availableSpace;

        if (!$enoughSpace) {
            return self::notEnoughSpaceMessage();
        }

        return null;
    }

    protected function validateAllowedExtensions(
        ?string $extension = null,
    ): string|null {
        if (
            $extension &&
            !empty($this->allowedExtensions) &&
            !$this->extensionMatches($extension, $this->allowedExtensions)
        ) {
            return __('Files of this type are not allowed');
        }

        return null;
    }

    protected function validateBlockedExtensions(
        ?string $extension = null,
        ?string $mime = null,
    ): string|null {
        if (
            $extension &&
            !empty($this->blockedExtensions) &&
            ($this->extensionMatches($extension, $this->blockedExtensions) ||
                $this->mimeMatches($mime, $this->blockedExtensions))
        ) {
            return __('Files of this type are not allowed');
        }

        return null;
    }

    protected function mimeMatches(string $mime, array $mimes): bool
    {
        if (empty($mimes)) {
            return false;
        }

        return in_array($mime, $mimes);
    }

    protected function extensionMatches(
        string $extension,
        array $extensions,
    ): bool {
        if (empty($extensions)) {
            return false;
        }

        $extensions = array_map(
            fn($ext) => str_replace('.', '', $ext),
            $extensions,
        );

        return in_array(str_replace('.', '', $extension), $extensions);
    }

    public static function formatBytes(?int $bytes, $unit = 'MB'): string
    {
        if (is_null($bytes)) {
            return '0 bytes';
        }

        if ((!$unit && $bytes >= 1 << 30) || $unit == 'GB') {
            return number_format($bytes / (1 << 30), 1) . 'GB';
        }
        if ((!$unit && $bytes >= 1 << 20) || $unit == 'MB') {
            return number_format($bytes / (1 << 20), 1) . 'MB';
        }
        if ((!$unit && $bytes >= 1 << 10) || $unit == 'KB') {
            return number_format($bytes / (1 << 10), 1) . 'KB';
        }
        return number_format($bytes) . ' bytes';
    }

    public static function notEnoughSpaceMessage(): string
    {
        return __(
            'You have exhausted your allowed space of :space. Delete some files or upgrade your plan.',
            [
                'space' => self::formatBytes(
                    app(GetUserSpaceUsage::class)->getAvailableSpace(),
                ),
            ],
        );
    }
}
