<?php

namespace Common\Files;

use Exception;
use Illuminate\Contracts\Filesystem\Filesystem;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Storage;

// todo: make sure visibility is set correctly when uploading a file based on upload type visibility (can refactor upload middlewares and set visibility, default config (file size, type etc) and other static config there)

class UploadDiskResolver
{
    public static function getUploadTypeConfig(string $uploadType): array
    {
        $userUploadTypeConfig =
            settings()->get("uploading.types.$uploadType") ?? [];
        $defaultUploadTypeConfig =
            config("filesystems.upload_types.$uploadType") ?? [];
        $mergedUploadTypeConfig = array_merge_recursive(
            $defaultUploadTypeConfig,
            $userUploadTypeConfig,
        );

        throw_if(
            empty($mergedUploadTypeConfig),
            new Exception("No configuration found for $uploadType"),
        );

        return $mergedUploadTypeConfig;
    }

    public static function getBackendConfig(string $backendId): array
    {
        return Arr::first(
            settings()->get('uploading.backends') ?? [],
            fn($backend) => $backend['id'] === $backendId,
        );
    }

    public static function getUploadTypeVisibility(string $uploadType): string
    {
        return Arr::get(
            self::getUploadTypeConfig($uploadType),
            'visibility',
            'private',
        );
    }

    public static function getAllUploadTypesConfig(): array
    {
        $config = [];

        foreach (
            array_keys(config('filesystems.upload_types') ?? [])
            as $uploadType
        ) {
            $config[$uploadType] = self::getUploadTypeConfig($uploadType);
        }

        return $config;
    }

    public function resolve(
        string $uploadType,
        ?string $backendId = null,
    ): Filesystem {
        $mergedUploadTypeConfig = self::getUploadTypeConfig($uploadType);

        if ($backendId) {
            $backendConfig = self::getBackendConfig($backendId);
        } else {
            // todo: this will not work due to differences in s3, s3 direct, tus and axios, need to send backend id from frontend
            $typeBackendId = Arr::random(
                $mergedUploadTypeConfig['backends'] ?? [],
            );
            $backendConfig = self::getBackendConfig($typeBackendId);
        }

        throw_if(
            !$backendConfig,
            new Exception(
                "No configured uploading backend found for $uploadType",
            ),
        );

        return $this->resolveFromConfig(
            $backendConfig,
            $mergedUploadTypeConfig,
        );
    }

    public function resolveFromConfig(
        array $backendConfig,
        array $uploadTypeConfig,
    ): Filesystem {
        $driver = $this->typeToDriver($backendConfig['type']);

        $backendRoot = $backendConfig['root'] ?? '';
        $uploadTypeRoot = $uploadTypeConfig['root'] ?? '';

        // upload type root only specifies a folder relative to
        // backend root, so absolute path is not supported here
        $uploadTypeRoot = $this->isAbsolutePath($uploadTypeRoot)
            ? ''
            : $uploadTypeRoot;

        if ($driver === 'local') {
            $localRoot = $this->getLocalRoot($uploadTypeConfig['visibility']);
            $backendRoot = $this->isAbsolutePath($backendRoot)
                ? $backendRoot
                : "$localRoot/$backendRoot";
        } else {
            $backendRoot = $this->isAbsolutePath($backendRoot)
                ? 'uploads'
                : $backendRoot;
        }

        return Storage::build([
            'driver' => $driver,
            'throw' => true,
            'root' => '',
            'prefix' => trim("$backendRoot/$uploadTypeRoot", '/'),
            ...$backendConfig['credentials'],
        ]);
    }

    protected function getLocalRoot(string $visibility): string
    {
        return $visibility === 'public'
            ? public_path('storage')
            : storage_path('app/uploads');
    }

    protected function typeToDriver(string $type): string
    {
        return match ($type) {
            'other_s3', 'digitalocean', 'backblaze' => 's3',
            default => $type,
        };
    }

    protected function isAbsolutePath(string $path): bool
    {
        // Handle empty path
        if (empty($path)) {
            return false;
        }

        // Windows absolute paths
        // C:\path, D:\path, etc. (drive letter followed by colon and backslash)
        if (isset(parse_url($path)['scheme'])) {
            return true;
        }

        if ($path[0] === '/' || $path[0] === '\\') {
            return true;
        }

        return false;
    }
}
